let currentVideoInfo = null;

function showError(message) {
    const errorDiv = document.getElementById('errorMessage');
    const successDiv = document.getElementById('successMessage');
    const videoInfo = document.getElementById('videoInfo');
    
    errorDiv.textContent = message;
    errorDiv.style.display = 'block';
    successDiv.style.display = 'none';
    videoInfo.style.display = 'none';
    
    setTimeout(() => {
        errorDiv.style.display = 'none';
    }, 5000);
}

function showSuccess(message) {
    const successDiv = document.getElementById('successMessage');
    const errorDiv = document.getElementById('errorMessage');
    
    successDiv.textContent = message;
    successDiv.style.display = 'block';
    errorDiv.style.display = 'none';
    
    setTimeout(() => {
        successDiv.style.display = 'none';
    }, 3000);
}

function setLoading(isLoading) {
    const fetchBtn = document.getElementById('fetchBtn');
    const btnText = document.getElementById('btnText');
    const btnLoader = document.getElementById('btnLoader');
    
    fetchBtn.disabled = isLoading;
    btnText.style.display = isLoading ? 'none' : 'inline';
    btnLoader.style.display = isLoading ? 'inline-block' : 'none';
}

async function fetchVideoInfo() {
    const urlInput = document.getElementById('twitterUrl');
    const url = urlInput.value.trim();
    
    if (!url) {
        showError('Please enter a Twitter/X URL');
        return;
    }
    
    // Basic URL validation
    if (!url.includes('twitter.com') && !url.includes('x.com')) {
        showError('Please enter a valid Twitter/X URL');
        return;
    }
    
    setLoading(true);
    
    try {
        const response = await fetch('/api/get-video-info', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ url: url })
        });
        
        const data = await response.json();
        
        if (!response.ok) {
            throw new Error(data.error || 'Failed to fetch video information');
        }
        
        currentVideoInfo = data;
        displayVideoInfo(data);
        
    } catch (error) {
        showError(error.message);
    } finally {
        setLoading(false);
    }
}

function displayVideoInfo(info) {
    const videoInfoDiv = document.getElementById('videoInfo');
    const videoTitle = document.getElementById('videoTitle');
    const errorDiv = document.getElementById('errorMessage');
    
    videoTitle.textContent = `Title: ${info.title}`;
    videoInfoDiv.style.display = 'block';
    errorDiv.style.display = 'none';
}

async function downloadVideo() {
    if (!currentVideoInfo) {
        showError('Please fetch video information first');
        return;
    }
    
    const downloadBtn = document.getElementById('downloadBtn');
    downloadBtn.disabled = true;
    downloadBtn.innerHTML = '<span class="loader"></span> Downloading...';
    
    try {
        const response = await fetch('/api/download', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                video_url: currentVideoInfo.video_url,
                filename: currentVideoInfo.filename
            })
        });
        
        if (!response.ok) {
            const data = await response.json();
            throw new Error(data.error || 'Failed to download video');
        }
        
        // Create a blob from the response
        const blob = await response.blob();
        
        // Create a temporary URL for the blob
        const url = window.URL.createObjectURL(blob);
        
        // Create a temporary anchor element and trigger download
        const a = document.createElement('a');
        a.style.display = 'none';
        a.href = url;
        a.download = currentVideoInfo.filename;
        document.body.appendChild(a);
        a.click();
        
        // Clean up
        window.URL.revokeObjectURL(url);
        document.body.removeChild(a);
        
        showSuccess('Video downloaded successfully!');
        
    } catch (error) {
        showError(error.message);
    } finally {
        downloadBtn.disabled = false;
        downloadBtn.innerHTML = `
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
                <polyline points="7 10 12 15 17 10"></polyline>
                <line x1="12" y1="15" x2="12" y2="3"></line>
            </svg>
            Download Video
        `;
    }
}

// Allow Enter key to trigger fetch
document.addEventListener('DOMContentLoaded', function() {
    const urlInput = document.getElementById('twitterUrl');
    urlInput.addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            fetchVideoInfo();
        }
    });
});
